// /////////////////////////////////////////////////////////////////////////////
// DR DOBB'S CHALLENGES
//
// Filename       : level.cpp
// Date           : February 2008
//
// Description    : Refer to description in corresponding header.
//
// ///////////////////////////////////////////////////////////////////////////


#include <tchar.h>

#include "level.h"

#include "entity.h"
#include "EntityParticle.h"

#include "Util.h"

#include "Application.h"



Level::Level( Game* pGame ) :
  m_Number( 0 ),
  m_Width( Dobbs::LEVEL_DEFAULT_WIDTH ),
  m_Height( Dobbs::LEVEL_DEFAULT_HEIGHT ),
  m_Tiles( NULL ),
  m_BackgroundTiles( NULL ),
  m_ForegroundTiles( NULL ),
  m_Tokens( 0 ),
  m_pGame( pGame ),
  m_WibbleProgress( 0.0f )
{

  m_DisplayLayer[0] = true;
  m_DisplayLayer[1] = true;
  m_DisplayLayer[2] = true;

  Clear();

  m_Tiles = new Dobbs::Tile[m_Width * m_Height];
  for ( int i = 0; i < m_Width * m_Height; ++i )
  {
    m_Tiles[i] = Dobbs::Tile();
  }
  m_BackgroundTiles = new Dobbs::TileType[m_Width * m_Height];
  for ( int i = 0; i < m_Width * m_Height; ++i )
  {
    m_BackgroundTiles[i] = Dobbs::TILE_EMPTY;
  }
  m_ForegroundTiles = new Dobbs::TileType[m_Width * m_Height];
  for ( int i = 0; i < m_Width * m_Height; ++i )
  {
    m_ForegroundTiles[i] = Dobbs::TILE_EMPTY;
  }

}



Level::~Level() 
{

  Clear();

}



void Level::Clear()
{

  m_Tokens    = 0;
  m_pPlayer   = NULL;

  m_Name.erase();

  m_Width   = Dobbs::LEVEL_DEFAULT_WIDTH;
  m_Height  = Dobbs::LEVEL_DEFAULT_HEIGHT;

  delete[] m_Tiles;
  m_Tiles = NULL;
  delete[] m_BackgroundTiles;
  m_BackgroundTiles = NULL;
  delete[] m_ForegroundTiles;
  m_ForegroundTiles = NULL;

  ClearEntities();

  for ( int i = 0; i < 4; ++i )
  {
    m_Keys[i] = false;
  }

  memset( &m_WibbleArray[0], 0, sizeof( m_WibbleArray ) );

  m_OffsetX = 0;
  m_OffsetY = 0;

}



void Level::ClearEntities()
{

  m_pPlayer = NULL;

  std::list<Entity*>::iterator    it( m_Entities.begin() );
  while ( it != m_Entities.end() )
  {
    delete *it;

    ++it;
  }
  m_Entities.clear();

}



void Level::initBasic() 
{

  // When a level file does not exist or is invalid, create a basic level
  // with just a platform on the bottom row, the player, and one token.

  // Clear existing tile array
  Clear();
  
  m_Tiles = new Dobbs::Tile[m_Width * m_Height];
  for ( int i = 0; i < m_Width * m_Height; ++i )
  {
    m_Tiles[i] = Dobbs::Tile();
  }
  m_BackgroundTiles = new Dobbs::TileType[m_Width * m_Height];
  for ( int i = 0; i < m_Width * m_Height; ++i )
  {
    m_BackgroundTiles[i] = Dobbs::TILE_EMPTY;
  }
  m_ForegroundTiles = new Dobbs::TileType[m_Width * m_Height];
  for ( int i = 0; i < m_Width * m_Height; ++i )
  {
    m_ForegroundTiles[i] = Dobbs::TILE_EMPTY;
  }

  // Create platform at bottom of screen
  for ( int x = 0; x < m_Width; x++ ) 
  {
    SetTile( 1, x, m_Height - 1, Dobbs::TILE_GRASS );
  }
  SpawnEntity( Dobbs::ENTITY_TYPE_TOKEN, 0, 4 * 32, ( m_Height - 2 ) * 32 );
  SpawnEntity( Dobbs::ENTITY_TYPE_PLAYER, 0, 20 * 32, ( m_Height - 1 ) * 32 - 1 );

}



void Level::WakeUpOrSleep( Entity* pEntity )
{

  const int WakeupDistanceX = 240;
  const int WakeupDistanceY = 180;
  const int SleepDistanceX = 320;
  const int SleepDistanceY = 240;

  int     X = pEntity->GetX() - m_OffsetX;
  int     Y = pEntity->GetY() - m_OffsetY;

  if ( ( X < -SleepDistanceX )
  ||   ( X > SleepDistanceX + Dobbs::SCREEN_WIDTH )
  ||   ( Y < -SleepDistanceY )
  ||   ( Y > SleepDistanceY + Dobbs::SCREEN_HEIGHT ) )
  {
    if ( ( !pEntity->m_Sleeping )
    &&   ( pEntity->m_DiesIfOutsideScreen ) )
    {
      pEntity->Die();
      return;
    }

    if ( ( !pEntity->m_Sleeping )
    &&   ( pEntity->m_CanSleep ) )
    {
      pEntity->m_Sleeping = true;
      pEntity->OnEvent( Entity::EE_SLEEP );
      return;
    }
  }
  if ( ( X > -WakeupDistanceX )
  &&   ( X < WakeupDistanceX + Dobbs::SCREEN_WIDTH )
  &&   ( Y > -WakeupDistanceY )
  &&   ( Y < WakeupDistanceY + Dobbs::SCREEN_HEIGHT ) )
  {
    if ( pEntity->m_Sleeping )
    {
      pEntity->m_Sleeping = false;
      pEntity->OnEvent( Entity::EE_WAKEUP );
    }
  }

}



void Level::Update()
{

  std::list<Entity*>::iterator   it( m_Entities.begin() );
  while ( it != m_Entities.end() )
  {
    Entity*   pEntity( *it );

    if ( pEntity->CanBeRemoved() )
    {
      pEntity->OnEvent( Entity::EE_DIE );
      delete pEntity;
      it = m_Entities.erase( it );
    }
    else
    {
      WakeUpOrSleep( pEntity );
      if ( !pEntity->m_Sleeping )
      {
        pEntity->Update( *this );
      }
      ++it;
    }
  }

}



bool Level::CanCollide( Dobbs::EntityTypes Type )
{

  if ( ( Type == Dobbs::ENTITY_TYPE_PLAYER )
  ||   ( Type == Dobbs::ENTITY_TYPE_THROWN_TOKEN )
  ||   ( Type == Dobbs::ENTITY_TYPE_SUB )
  ||   ( Type == Dobbs::ENTITY_TYPE_SUB_SHOT ) )
  {
    return true;
  }
  return false;

}



bool Level::CanCollide( Dobbs::EntityTypes Type, Dobbs::EntityTypes Type2 )
{

  switch ( Type )
  {
    case Dobbs::ENTITY_TYPE_PLAYER:
    case Dobbs::ENTITY_TYPE_SUB:
      if ( ( Type2 == Dobbs::ENTITY_TYPE_TOKEN )
      ||   ( Type2 == Dobbs::ENTITY_TYPE_THROWN_TOKEN )
      ||   ( Type2 == Dobbs::ENTITY_TYPE_KEY_BLUE )
      ||   ( Type2 == Dobbs::ENTITY_TYPE_KEY_GREEN )
      ||   ( Type2 == Dobbs::ENTITY_TYPE_KEY_RED )
      ||   ( Type2 == Dobbs::ENTITY_TYPE_KEY_YELLOW )
      ||   ( Type2 == Dobbs::ENTITY_TYPE_SHODAN )
      ||   ( Type2 == Dobbs::ENTITY_TYPE_BUG_L )
      ||   ( Type2 == Dobbs::ENTITY_TYPE_BUG_R )
      ||   ( Type2 == Dobbs::ENTITY_TYPE_FISH_L )
      ||   ( Type2 == Dobbs::ENTITY_TYPE_FISH_R )
      ||   ( Type2 == Dobbs::ENTITY_TYPE_FLY )
      ||   ( Type2 == Dobbs::ENTITY_TYPE_CANNON )
      ||   ( Type2 == Dobbs::ENTITY_TYPE_SUB )
      ||   ( Type2 == Dobbs::ENTITY_TYPE_SECURITY_PASS )
      ||   ( Type2 == Dobbs::ENTITY_TYPE_LASER_H )
      ||   ( Type2 == Dobbs::ENTITY_TYPE_LASER_V )
      ||   ( Type2 == Dobbs::ENTITY_TYPE_SHOT )
      ||   ( Type2 == Dobbs::ENTITY_TYPE_CAR ) )
      {
        return true;
      }
      break;
    case Dobbs::ENTITY_TYPE_SUB_SHOT:
      if ( ( Type2 == Dobbs::ENTITY_TYPE_BUG_L )
      ||   ( Type2 == Dobbs::ENTITY_TYPE_BUG_R )
      ||   ( Type2 == Dobbs::ENTITY_TYPE_FISH_L )
      ||   ( Type2 == Dobbs::ENTITY_TYPE_FISH_R )
      ||   ( Type2 == Dobbs::ENTITY_TYPE_FLY )
      ||   ( Type2 == Dobbs::ENTITY_TYPE_SHODAN )
      ||   ( Type2 == Dobbs::ENTITY_TYPE_LASER_D )
      ||   ( Type2 == Dobbs::ENTITY_TYPE_LASER_U )
      ||   ( Type2 == Dobbs::ENTITY_TYPE_LASER_L )
      ||   ( Type2 == Dobbs::ENTITY_TYPE_LASER_R )
      ||   ( Type2 == Dobbs::ENTITY_TYPE_ANOMALY ) )
      {
        return true;
      }
      break;
    case Dobbs::ENTITY_TYPE_THROWN_TOKEN:
      if ( ( Type2 == Dobbs::ENTITY_TYPE_BUG_L )
      ||   ( Type2 == Dobbs::ENTITY_TYPE_BUG_R )
      ||   ( Type2 == Dobbs::ENTITY_TYPE_FLY )
      ||   ( Type2 == Dobbs::ENTITY_TYPE_SHODAN )
      ||   ( Type2 == Dobbs::ENTITY_TYPE_FISH_L )
      ||   ( Type2 == Dobbs::ENTITY_TYPE_FISH_R )
      ||   ( Type2 == Dobbs::ENTITY_TYPE_ANOMALY ) )
      {
        return true;
      }
      break;
  }
  return false;

}



void Level::UpdateTimed( float ElapsedTime )
{

  m_WibbleProgress += ElapsedTime;

  std::list<Entity*>::iterator   it( m_Entities.begin() );
  while ( it != m_Entities.end() )
  {
    Entity*   pEntity( *it );

    if ( pEntity->CanBeRemoved() )
    {
      pEntity->OnEvent( Entity::EE_DIE );
      delete pEntity;
      it = m_Entities.erase( it );
    }
    else
    {
      WakeUpOrSleep( pEntity );
      if ( !pEntity->m_Sleeping )
      {
        pEntity->UpdateTimed( *this, ElapsedTime );
        // check agains so we don't display dead entites
        if ( pEntity->CanBeRemoved() )
        {
          pEntity->OnEvent( Entity::EE_DIE );
          delete pEntity;
          it = m_Entities.erase( it );
          continue;
        }
      }

      ++it;
    }
  }

  it = m_Entities.begin();
  while ( it != m_Entities.end() )
  {
    Entity*   pEntity( *it );

    if ( ( !pEntity->CanBeRemoved() )
    &&   ( !pEntity->IsHidden() )
    &&   ( CanCollide( pEntity->GetType() ) ) )
    {
      std::list<Entity*>::iterator   it2( m_Entities.begin() );
      while ( it2 != m_Entities.end() )
      {
        Entity*   pEntity2( *it2 );

        if ( pEntity->CanBeRemoved() )
        {
          break;
        }
        if ( ( !pEntity2->CanBeRemoved() )
        &&   ( !pEntity2->IsHidden() ) )
        {
          if ( ( CanCollide( pEntity->GetType(), pEntity2->GetType() ) )
          &&   ( Collide( pEntity, pEntity2 ) ) )
          {
            if ( m_pGame )
            {
              m_pGame->OnCollision( pEntity, pEntity2 );
              m_pGame->OnCollision( pEntity2, pEntity );
            }
          }
        }
        ++it2;
      }
    }
    ++it;
  }

}



void Level::InsertEntity( Entity* pEntity )
{

  std::list<Entity*>::iterator    it( m_Entities.begin() );
  while ( it != m_Entities.end() )
  {
    Entity* pOtherEntity( *it );

    if ( pOtherEntity->m_Z < pEntity->m_Z )
    {
      m_Entities.insert( it, pEntity );
      return;
    }

    ++it;
  }

  m_Entities.push_back( pEntity );

}



Entity* Level::SpawnEntity( const Dobbs::EntityTypes EntityType, int StartDir, int X, int Y )
{

  Entity*   pEntity = Entity::SpawnEntity( EntityType, X, Y, (Dobbs::DirFlags)StartDir );
  if ( pEntity == NULL )
  {
    return NULL;
  }
  pEntity->m_pLevel = this;
  pEntity->OnEvent( Entity::EE_INIT );

  InsertEntity( pEntity );
  
  return pEntity;

}



void Level::SetTile( int Layer, int X, int Y, Dobbs::TileType NewTile )
{

  if ( ( X < 0 )
  ||   ( X >= m_Width )
  ||   ( Y < 0 )
  ||   ( Y >= m_Height ) )
  {
    return;
  }
  if ( NewTile >= Dobbs::TILE_LAST_ENTRY )
  {
    NewTile = Dobbs::TILE_EMPTY;
  }

  switch ( Layer )
  {
    case 0:
      m_BackgroundTiles[X + m_Width * Y] = NewTile;
      break;
    case 1:
      m_Tiles[X + m_Width * Y].Type = NewTile;
      m_Tiles[X + m_Width * Y].Color = 0xffffffff;
      break;
    case 2:
      m_ForegroundTiles[X + m_Width * Y] = NewTile;
      break;
  }

}



Dobbs::TileType Level::Tile( int tx, int ty )
{

  if ( ( tx < 0 )
  ||   ( tx >= m_Width )
  ||   ( ty < 0 )
  ||   ( ty >= m_Height ) )
  {
    // dummy tile for out of border access
    return Dobbs::TILE_EMPTY;
  }
  return m_Tiles[tx + ty * m_Width].Type;

}



Dobbs::Tile& Level::FullTile( int tx, int ty )
{

  if ( ( tx < 0 )
  ||   ( tx >= m_Width )
  ||   ( ty < 0 )
  ||   ( ty >= m_Height ) )
  {
    // dummy tile for out of border access
    static Dobbs::Tile    DummyTile;

    return DummyTile;
  }
  return m_Tiles[tx + ty * m_Width];

}



Dobbs::TileType Level::TileFromLayer( int Layer, int tx, int ty )
{

  if ( ( tx < 0 )
  ||   ( tx >= m_Width )
  ||   ( ty < 0 )
  ||   ( ty >= m_Height ) )
  {
    // dummy tile for out of border access
    return Dobbs::TILE_EMPTY;
  }
  switch ( Layer )
  {
    case 0:
      return m_BackgroundTiles[tx + ty * m_Width];
    case 1:
      return m_Tiles[tx + ty * m_Width].Type;
    case 2:
      return m_ForegroundTiles[tx + ty * m_Width];
  }
  return Dobbs::TILE_EMPTY;

}



DWORD Level::TileColor( int tx, int ty )
{

  if ( ( tx < 0 )
  ||   ( tx >= m_Width )
  ||   ( ty < 0 )
  ||   ( ty >= m_Height ) )
  {
    // dummy tile for out of border access
    return 0xffffffff;
  }
  return m_Tiles[tx + ty * m_Width].Color;

}



bool Level::Save( int Nr )
{

  FILE*     ioOut = fopen( Util::AppPath( "levels\\level%03d.lev", Nr ), "wb" );
  if ( ioOut == NULL )
  {
    return false;
  }
  int   Width = m_Width;
  fwrite( &Width, sizeof( Width ), 1, ioOut );
  int   Height = m_Height;
  fwrite( &Height, sizeof( Height ), 1, ioOut );

  fwrite( m_Tiles, sizeof( Dobbs::Tile ) * m_Width * m_Height, 1, ioOut );

  size_t    EntityCount = m_Entities.size();
  fwrite( &EntityCount, sizeof( EntityCount ), 1, ioOut );
  std::list<Entity*>::iterator   it( m_Entities.begin() );
  while ( it != m_Entities.end() )
  {
    Entity*   pEntity( *it );

    int   Type = pEntity->GetType();
    int   X = pEntity->GetX();
    int   Y = pEntity->GetY();
    int   ExtraData = 0;

    fwrite( &Type, sizeof( Type ), 1, ioOut );
    fwrite( &X, sizeof( X ), 1, ioOut );
    fwrite( &Y, sizeof( Y ), 1, ioOut );
    // reserved data
    fwrite( &pEntity->m_ExtraData, sizeof( pEntity->m_ExtraData ), 1, ioOut );
    fwrite( &pEntity->m_ExtraData2, sizeof( pEntity->m_ExtraData2 ), 1, ioOut );
    fwrite( &pEntity->m_Angle, sizeof( pEntity->m_Angle ), 1, ioOut );
    fwrite( &ExtraData, sizeof( ExtraData ), 1, ioOut );

    ++it;
  }

  // Level name
  int   TextLength = (int)m_Name.length();
  fwrite( &TextLength, sizeof( TextLength ), 1, ioOut );
  fwrite( m_Name.c_str(), TextLength, 1, ioOut );

  fwrite( m_BackgroundTiles, sizeof( Dobbs::TileType ) * m_Width * m_Height, 1, ioOut );
  fwrite( m_ForegroundTiles, sizeof( Dobbs::TileType ) * m_Width * m_Height, 1, ioOut );

  fclose( ioOut );

  return true;

}



bool Level::Load( int Nr )
{

  Clear();
  m_Number = Nr;

  FILE*     ioIn = fopen( Util::AppPath( "levels\\level%03d.lev", Nr ), "rb" );
  if ( ioIn == NULL )
  {
    initBasic();
    return false;
  }
  fread( &m_Width, sizeof( m_Width ), 1, ioIn );
  fread( &m_Height, sizeof( m_Height ), 1, ioIn );

  m_Tiles = new Dobbs::Tile[m_Width * m_Height];
  m_BackgroundTiles = new Dobbs::TileType[m_Width * m_Height];
  m_ForegroundTiles = new Dobbs::TileType[m_Width * m_Height];
  fread( m_Tiles, sizeof( Dobbs::Tile ) * m_Width * m_Height, 1, ioIn );

  /*
  for ( int i = 0; i < m_Width; ++i )
  {
    for ( int j = 0; j < m_Height; ++j )
    {
      Dobbs::TileType   aTile;

      fread( &aTile, sizeof( aTile ), 1, ioIn );
      SetTile( i, j, aTile );
    }
  }
  */

  size_t    EntityCount = 0;

  fread( &EntityCount, sizeof( EntityCount ), 1, ioIn );

  for ( size_t i = 0; i < EntityCount; ++i )
  {
    int   Type;
    int   X;
    int   Y;
    int   ExtraData = 0;
    int   ExtraData2 = 0;
    float ExtraData3 = 0;
    int   ExtraData4 = 0;

    fread( &Type, sizeof( Type ), 1, ioIn );
    fread( &X, sizeof( X ), 1, ioIn );
    fread( &Y, sizeof( Y ), 1, ioIn );
    // reserved data
    fread( &ExtraData, sizeof( ExtraData ), 1, ioIn );
    fread( &ExtraData2, sizeof( ExtraData2 ), 1, ioIn );
    fread( &ExtraData3, sizeof( ExtraData3 ), 1, ioIn );
    fread( &ExtraData4, sizeof( ExtraData4 ), 1, ioIn );

    Entity* pEntity = SpawnEntity( (Dobbs::EntityTypes)Type, ExtraData, X, Y );

    pEntity->m_ExtraData = ExtraData;
    pEntity->m_ExtraData2 = ExtraData2;
    pEntity->SetAngle( ExtraData3 );

    pEntity->OnEvent( Entity::EE_INIT );
    pEntity->m_Sleeping = true;
  }

  // Level name
  int   TextLength = 0;
  fread( &TextLength, sizeof( TextLength ), 1, ioIn );
  char*   pTemp = new char[TextLength];
  fread( pTemp, TextLength, 1, ioIn );
  m_Name.append( pTemp, TextLength );
  delete[] pTemp;

  fread( m_BackgroundTiles, sizeof( Dobbs::TileType ) * m_Width * m_Height, 1, ioIn );
  fread( m_ForegroundTiles, sizeof( Dobbs::TileType ) * m_Width * m_Height, 1, ioIn );

  fclose( ioIn );

  for ( int j = 0; j < m_Height; ++j )
  {
    for ( int i = 0; i < m_Width; ++i )
    {
      if ( TileFromLayer( 0, i, j ) == 0xcdcdcdcd )
      {
        SetTile( 0, i, j, Dobbs::TILE_EMPTY );
      }
      if ( TileFromLayer( 2, i, j ) == 0xcdcdcdcd )
      {
        SetTile( 2, i, j, Dobbs::TILE_EMPTY );
      }
    }
  }

  return true;

}



int Level::Width()
{

  return m_Width;

}



int Level::Height()
{

  return m_Height;

}



void Level::TokenCollected()
{

  --m_Tokens;

}



bool Level::IsDone()
{

  return ( m_Tokens == 0 );

}



void Level::DeleteSpawnedEntityAt( int x, int y )
{

  std::list<Entity*>::iterator    it( m_Entities.begin() );
  while ( it != m_Entities.end() )
  {
    Entity* pEntity( *it );

    if ( ( pEntity->m_SpawnPosX == x )
    &&   ( pEntity->m_SpawnPosY == y ) )
    {
      delete pEntity;
      m_Entities.erase( it );
      break;
    }

    ++it;
  }

}



void Level::Resize( int NewWidth, int NewHeight )
{

  if ( ( NewWidth == m_Width )
  &&   ( NewHeight == m_Height ) )
  {
    return;
  }
  if ( ( NewWidth <= 0 )
  ||   ( NewHeight <= 0 ) )
  {
    return;
  }

  Dobbs::Tile*    pNewTiles = new Dobbs::Tile[NewWidth * NewHeight];
  Dobbs::TileType*    pNewTilesBG = new Dobbs::TileType[NewWidth * NewHeight];
  Dobbs::TileType*    pNewTilesFG = new Dobbs::TileType[NewWidth * NewHeight];

  for ( int i = 0; i < NewWidth; ++i )
  {
    for ( int j = 0; j < NewHeight; ++j )
    {
      if ( ( i < m_Width )
      &&   ( j < m_Height ) )
      {
        // copy old tiles over
        pNewTiles[i + j * NewWidth] = m_Tiles[i + j * m_Width];
        pNewTilesBG[i + j * NewWidth] = m_BackgroundTiles[i + j * m_Width];
        pNewTilesFG[i + j * NewWidth] = m_ForegroundTiles[i + j * m_Width];
      }
      else
      {
        pNewTiles[i + j * NewWidth] = Dobbs::TILE_EMPTY;
        pNewTilesBG[i + j * NewWidth] = Dobbs::TILE_EMPTY;
        pNewTilesFG[i + j * NewWidth] = Dobbs::TILE_EMPTY;
      }
    }
  }
  delete[] m_Tiles;
  delete[] m_BackgroundTiles;
  delete[] m_ForegroundTiles;

  m_Tiles           = pNewTiles;
  m_BackgroundTiles = pNewTilesBG;
  m_ForegroundTiles = pNewTilesFG;
  m_Width   = NewWidth;
  m_Height  = NewHeight;

}



Entity* Level::FindSpawnedEntityFrom( int X, int Y )
{

  std::list<Entity*>::iterator    it( m_Entities.begin() );
  while ( it != m_Entities.end() )
  {
    Entity* pEntity( *it );

    if ( ( pEntity->m_SpawnPosX == X )
    &&   ( pEntity->m_SpawnPosY == Y ) )
    {
      return pEntity;
    }

    ++it;
  }
  return NULL;

}



Entity* Level::FindCollidingEntity( Entity* pEntity, Dobbs::EntityTypes Type )
{

  // find an entity that's colliding with the passed entity and has the proper type
  std::list<Entity*>::iterator    it( m_Entities.begin() );
  while ( it != m_Entities.end() )
  {
    Entity* pOtherEntity( *it );

    if ( pOtherEntity->GetType() == Type )
    {
      if ( pEntity->CollidesWith( pOtherEntity ) )
      {
        return pOtherEntity;
      }
    }
    ++it;
  }
  return NULL;

}


void Level::UpdateWibbleArray()
{

  memset( &m_WibbleArray, 0, sizeof( m_WibbleArray ) );

  for ( int i = -1; i < ( Dobbs::SCREEN_WIDTH / 32 ) + 2; ++i )
  {
    for ( int j = -1; j < ( Dobbs::SCREEN_HEIGHT / 32 ) + 2; ++j )
    {
      Dobbs::TileType   TLO = TileFromLayer( 2, m_OffsetX / 32 + i, m_OffsetY / 32 + j );
      Dobbs::TileType   TRO = TileFromLayer( 2, m_OffsetX / 32 + i + 1, m_OffsetY / 32 + j );
      Dobbs::TileType   TLU = TileFromLayer( 2, m_OffsetX / 32 + i, m_OffsetY / 32 + j + 1 );
      Dobbs::TileType   TRU = TileFromLayer( 2, m_OffsetX / 32 + i + 1, m_OffsetY / 32 + j + 1 );

      if ( ( TRU == Dobbs::TILE_WATER )
      &&   ( TLO == TRO )
      &&   ( TRO == TLU )
      &&   ( TLU == TRU ) )
      {
        m_WibbleArray[i + 2][j + 2][0] = (int)( 5.0f * sinf( ( 130.0f * m_WibbleProgress + 50 * ( m_OffsetX / 32 + m_OffsetY / 32 + i + j ) ) * 3.1415926f / 180.0f ) );
        m_WibbleArray[i + 2][j + 2][1] = (int)( 5.0f * cosf( ( 130.0f * m_WibbleProgress + 50 * ( m_OffsetX / 32 + m_OffsetY / 32 + i + j ) ) * 3.1415926f / 180.0f ) );
      }
    }
  }

}



void Level::Render( DWORD ModulateColorArg )
{

  UpdateWibbleArray();

  int     DX = m_OffsetX % 32;
  int     DY = m_OffsetY % 32;

  for ( int i = -1; i < ( Dobbs::SCREEN_WIDTH / 32 ) + 2; ++i )
  {
    for ( int j = -1; j < ( Dobbs::SCREEN_HEIGHT / 32 ) + 2; ++j )
    {
      int     TX = m_OffsetX / 32 + i;
      int     TY = m_OffsetY / 32 + j;
      Dobbs::TileType       bgTile( TileFromLayer( 0, TX, TY ) );
      const Dobbs::Tile&    aTile( FullTile( TX, TY ) );

      unsigned int          Color( Util::ModulateColor( ModulateColorArg, aTile.Color ) );

      if ( m_DisplayLayer[0] )
      {
        if ( bgTile != Dobbs::TILE_EMPTY )
        {
          g_App.RenderTile( bgTile, 
                            i * 32 - DX + m_WibbleArray[i + 1][j + 1][0], j * 32 - DY + m_WibbleArray[i + 1][j + 1][1], 
                            i * 32 - DX + m_WibbleArray[i + 2][j + 1][0] + 32, j * 32 - DY + m_WibbleArray[i + 2][j + 1][1], 
                            i * 32 - DX + m_WibbleArray[i + 1][j + 2][0], j * 32 - DY + m_WibbleArray[i + 1][j + 2][1] + 32, 
                            i * 32 - DX + m_WibbleArray[i + 2][j + 2][0] + 32, j * 32 - DY + m_WibbleArray[i + 2][j + 2][1] + 32,
                            Color );
        }
      }
    }
  }
  if ( m_DisplayLayer[1] )
  {
    for ( int i = -1; i < ( Dobbs::SCREEN_WIDTH / 32 ) + 2; ++i )
    {
      for ( int j = -1; j < ( Dobbs::SCREEN_HEIGHT / 32 ) + 2; ++j )
      {
        int     TX = m_OffsetX / 32 + i;
        int     TY = m_OffsetY / 32 + j;
        const Dobbs::Tile&    aTile( FullTile( TX, TY ) );

        if ( aTile.Type != Dobbs::TILE_EMPTY )
        {
          //g_App.RenderTile( aTile.Type, i * 32 - DX + m_WibbleArray[i + 1][j + 1][0], j * 32 - DY + m_WibbleArray[i + 1][j + 1][1], aTile.Color );
          g_App.RenderTile( aTile.Type, 
                            i * 32 - DX, j * 32 - DY, 
                            i * 32 - DX + 32, j * 32 - DY, 
                            i * 32 - DX, j * 32 - DY + 32, 
                            i * 32 - DX + 32, j * 32 - DY + 32, 
                            ModulateColorArg );
        }
      }
    }
  }


  std::list<Entity*>::iterator    it( m_Entities.begin() );
  while ( it != m_Entities.end() )
  {
    Entity*   pEntity( *it );

    pEntity->Render( m_OffsetX, m_OffsetY );

    ++it;
  }

  if ( m_DisplayLayer[2] )
  {
    for ( int i = -1; i < ( Dobbs::SCREEN_WIDTH / 32 ) + 2; ++i )
    {
      for ( int j = -1; j < ( Dobbs::SCREEN_HEIGHT / 32 ) + 2; ++j )
      {
        int     TX = m_OffsetX / 32 + i;
        int     TY = m_OffsetY / 32 + j;
        Dobbs::TileType       fgTile( TileFromLayer( 2, TX, TY ) );
        if ( fgTile != Dobbs::TILE_EMPTY )
        {
          g_App.RenderTile( fgTile, 
                            i * 32 - DX, j * 32 - DY, 
                            i * 32 - DX + 32, j * 32 - DY, 
                            i * 32 - DX, j * 32 - DY + 32, 
                            i * 32 - DX + 32, j * 32 - DY + 32, 
                            ModulateColorArg );
        }
      }
    }
  }

}



bool Level::IsTileBlocking( Entity* pEntity, Dobbs::TileType Tile, Dobbs::TileType FGTile, int DirFlags )
{

  if ( ( pEntity )
  &&   ( pEntity->GetType() == Dobbs::ENTITY_TYPE_SUB ) )
  {
    if ( ( FGTile != Dobbs::TILE_WATER )
    &&   ( FGTile != Dobbs::TILE_WATER_TOP ) )
    {
      return true;
    }
  }

  if ( ( Tile == Dobbs::TILE_EMPTY )
  ||   ( Tile == Dobbs::TILE_SIGN )
  ||   ( Tile == Dobbs::TILE_SCREEN_SIGN )
  ||   ( Tile == Dobbs::TILE_METAL_DOOR_OPEN )
  ||   ( Tile == Dobbs::TILE_DOOR_OPENED )
  ||   ( Tile == Dobbs::TILE_LOCKDOWN_OPEN )
  ||   ( Tile == Dobbs::TILE_A )
  ||   ( Tile == Dobbs::TILE_B )
  ||   ( Tile == Dobbs::TILE_C )
  ||   ( Tile == Dobbs::TILE_D )
  ||   ( Tile == Dobbs::TILE_1 )
  ||   ( Tile == Dobbs::TILE_2 )
  ||   ( Tile == Dobbs::TILE_3 )
  ||   ( Tile == Dobbs::TILE_4 )
  ||   ( Tile == Dobbs::TILE_KELP_1 )
  ||   ( Tile == Dobbs::TILE_KELP_2 )
  ||   ( Tile == Dobbs::TILE_KELP_V_1 )
  ||   ( Tile == Dobbs::TILE_KELP_V_2 )
  ||   ( Tile == Dobbs::TILE_SPIKES_N )
  ||   ( Tile == Dobbs::TILE_SPIKES_E )
  ||   ( Tile == Dobbs::TILE_SPIKES_W )
  ||   ( Tile == Dobbs::TILE_SPIKES_S )
  ||   ( Tile == Dobbs::TILE_ARROW_LEFT )
  ||   ( Tile == Dobbs::TILE_ARROW_RIGHT )
  ||   ( Tile == Dobbs::TILE_WATER_TOP )
  ||   ( Tile == Dobbs::TILE_WATER )
  ||   ( Tile == Dobbs::TILE_PLATINE_NE )
  ||   ( Tile == Dobbs::TILE_PLATINE_NW ) )
  {
    return false;
  }
  if ( pEntity )
  {
    if ( pEntity->GetType() == Dobbs::ENTITY_TYPE_PLAYER )
    {
      // can pass from below
      if ( ( DirFlags != Dobbs::DIR_FALL )
      &&   ( ( Tile == Dobbs::TILE_GRASS )
      ||     ( Tile == Dobbs::TILE_TRAIN_TRACK )
      ||     ( Tile == Dobbs::TILE_METAL_PLATFORM )
      ||     ( Tile == Dobbs::TILE_GLUE ) ) )
      {
        return false;
      }
      if ( ( DirFlags == Dobbs::DIR_FALL )
      &&   ( Tile == Dobbs::TILE_TRAIN_TRACK ) )
      {
        return false;
      }
    }
    else if ( ( pEntity->GetType() == Dobbs::ENTITY_TYPE_FISH_L )
    ||        ( pEntity->GetType() == Dobbs::ENTITY_TYPE_FISH_R ) )
    {
      if ( ( DirFlags == Dobbs::DIR_FALL )
      ||   ( DirFlags == Dobbs::DIR_UP )
      ||   ( DirFlags == Dobbs::DIR_DOWN ) )
      {
        return false;
      }
    }
  }
  if ( ( DirFlags != Dobbs::DIR_FALL )
  &&   ( ( Tile == Dobbs::TILE_METAL_PLATFORM )
  ||     ( Tile == Dobbs::TILE_TRAIN_TRACK ) ) )
  {
    return false;
  }

  return true;

}



bool Level::IsTileDeadly( Entity* pEntity, Dobbs::TileType Tile, int DirFlags )
{

  if ( ( Tile == Dobbs::TILE_SPIKES_E )
  ||   ( Tile == Dobbs::TILE_SPIKES_W )
  ||   ( Tile == Dobbs::TILE_SPIKES_N )
  ||   ( Tile == Dobbs::TILE_SPIKES_S )
  ||   ( Tile == Dobbs::TILE_SPIKES_S )
  ||   ( Tile == Dobbs::TILE_SPIKE ) )
  {
    return true;
  }
  if ( ( Tile == Dobbs::TILE_WATER_TOP )
  ||   ( Tile == Dobbs::TILE_WATER ) )
  {
    if ( pEntity->GetType() == Dobbs::ENTITY_TYPE_PLAYER )
    {
      return true;
    }
    /*
    if ( ( pEntity->GetType() == Dobbs::ENTITY_TYPE_PLAYER )
    &&   ( ( (EntityPlayer*)pEntity )->GetState() != Dobbs::STATE_IN_SUB ) )
    {
      return true;
    }
    */
  }
  return false;

}



bool Level::BlockingEntity( Entity* pEntity1, Entity* pEntity2, int DirFlags )
{

  switch ( pEntity1->GetType() )
  {
    case Dobbs::ENTITY_TYPE_PLAYER:
      /*
      if ( pEntity2->GetType() == Dobbs::ENTITY_TYPE_PLATFORM )
      {
        if ( DirFlags & Dobbs::DIR_FALL )
        {
          return true;
        }
      }
      */
      break;
  }
  return false;

}



bool Level::BlockedByEntity( Entity* pEntity, RECT& rc, int DirFlags )
{

  std::list<Entity*>::iterator    it( m_Entities.begin() );
  while ( it != m_Entities.end() )
  {
    Entity*   pOtherEntity( *it );

    if ( ( pOtherEntity != pEntity )
    &&   ( BlockingEntity( pEntity, pOtherEntity, DirFlags ) ) )
    {
      RECT    rcTemp;
      if ( IntersectRect( &rcTemp, &rc, &pOtherEntity->CollisionRect() ) )
      {
        return true;
      }
    }

    ++it;
  }
  return false;

}



bool Level::IsPlatform( Entity* pEntity )
{

  if ( pEntity->GetType() == Dobbs::ENTITY_TYPE_PLATFORM )
  {
    return true;
  }
  return false;

}



bool Level::CheckPlatformBelow( Entity* pEntity )
{

  if ( !pEntity->CanBeCarried() )
  {
    return false;
  }
  RECT    rc = pEntity->CollisionRect();

  rc.top = rc.bottom;
  rc.bottom++;

  std::list<Entity*>::iterator    it( m_Entities.begin() );
  while ( it != m_Entities.end() )
  {
    Entity*   pOtherEntity( *it );

    if ( ( pOtherEntity != pEntity )
    &&   ( IsPlatform( pOtherEntity ) ) )
    {
      RECT    rcTemp;
      RECT    rcOther( pOtherEntity->CollisionRect() );
      // only the top border is valid for platforms
      rcOther.bottom = rcOther.top + 1;
      if ( IntersectRect( &rcTemp, &rc, &rcOther ) )
      {
        if ( pEntity->Carrier() != pOtherEntity )
        {
          pOtherEntity->Carry( pEntity );
          if ( !pEntity->m_OnGround )
          {
            pEntity->OnEvent( Entity::EE_LAND );
            //dh::Log( "Land e" );
            pEntity->m_FractY    = 0.0f;
            pEntity->m_JumpCount = 0;
            pEntity->m_FallCount = 0;
            pEntity->m_FallSpeed = 10;
            pEntity->m_OnGround = true;
          }
        }
        return true;
      }
    }

    ++it;
  }
  Entity*   pCarrier = pEntity->Carrier();
  if ( ( pCarrier )
  &&   ( IsPlatform( pCarrier ) ) )
  {
    pEntity->Carrier()->DropOff( pEntity );
  }
  return false;

}



bool Level::IsAreaBlocked( Entity* pEntity, RECT& rc, int DirFlags )
{


  int   X1 = rc.left / 32;
  int   Y1 = rc.top / 32;
  int   X2 = ( rc.right - 1 ) / 32;
  int   Y2 = ( rc.bottom - 1 ) / 32;

  if ( rc.left < 0 )
  {
    X1--;
  }
  if ( rc.top < 0 )
  {
    Y1--;
  }

  bool  Blocked = false;

  for ( int X = X1; X <= X2; ++X )
  {
    for ( int Y = Y1; Y <= Y2; ++Y )
    {
      if ( ( X < 0 )
      ||   ( X >= m_Width )
      ||   ( Y < 0 )
      ||   ( Y >= m_Height )
      ||   ( IsTileBlocking( pEntity, Tile( X, Y ), TileFromLayer( 2, X, Y ), DirFlags ) ) )
      {
        Blocked = true;
      }
    }
  }
  return Blocked;

}



bool Level::IsAreaDeadly( Entity* pEntity, RECT& rc, int DirFlags )
{


  int   X1 = rc.left / 32;
  int   Y1 = rc.top / 32;
  int   X2 = ( rc.right - 1 ) / 32;
  int   Y2 = ( rc.bottom - 1 ) / 32;

  if ( rc.left < 0 )
  {
    X1--;
  }
  if ( rc.top < 0 )
  {
    Y1--;
  }

  bool  Deadly = false;

  for ( int X = X1; X <= X2; ++X )
  {
    for ( int Y = Y1; Y <= Y2; ++Y )
    {
      if ( ( X < 0 )
      ||   ( X >= m_Width )
      ||   ( Y < 0 )
      ||   ( Y >= m_Height )
      ||   ( IsTileDeadly( pEntity, Tile( X, Y ), DirFlags ) )
      ||   ( IsTileDeadly( pEntity, TileFromLayer( 2, X, Y ), DirFlags ) ) )
      {
        Deadly = true;
      }
    }
  }
  return Deadly;

}



bool Level::CanMove( Entity* pEntity, int DX, int DY, int DirFlags )
{

  RECT    CollRect = pEntity->CollisionRect();
  RECT    CollRect2 = CollRect;

  if ( DY > 0 )
  {
    CollRect2.top = CollRect2.bottom;
    CollRect2.bottom++;
  }
  else if ( DY < 0 )
  {
    CollRect2.bottom = CollRect2.top;
    CollRect2.top--;
  }
  if ( DX > 0 )
  {
    CollRect2.left = CollRect2.right;
    CollRect2.right++;
  }
  else if ( DX < 0 )
  {
    CollRect2.right = CollRect2.left;
    CollRect2.left--;
  }

  //OffsetRect( &CollRect2, DX, DY );
  if ( BlockedByEntity( pEntity, CollRect2, DirFlags ) )
  {
    return true;
  }

  if ( ( ( DY > 0 )
  &&     ( ( CollRect.bottom % 32 ) != 0 ) )
  ||   ( ( DY < 0 )
  &&     ( ( CollRect.top % 32 ) != 0 ) )
  ||   ( ( DX > 0 )
  &&     ( ( CollRect.right % 32 ) != 0 ) )
  ||   ( ( DX < 0 )
  &&     ( ( CollRect.left % 32 ) != 0 ) ) )
  {
    // not blocked by tile
    return true;
  }

  if ( DY > 0 )
  {
    CollRect.top = CollRect.bottom;
    CollRect.bottom++;
  }
  else if ( DY < 0 )
  {
    CollRect.bottom = CollRect.top;
    CollRect.top--;
  }
  if ( DX > 0 )
  {
    CollRect.left = CollRect.right;
    CollRect.right++;
  }
  else if ( DX < 0 )
  {
    CollRect.right = CollRect.left;
    CollRect.left--;
  }
  //OffsetRect( &CollRect, DX, DY );

  return !IsAreaBlocked( pEntity, CollRect, DirFlags );

}



bool Level::Collide( Entity* pEntity, Entity* pEntity2 )
{

  if ( ( pEntity->CanBeRemoved() )
  ||   ( pEntity2->CanBeRemoved() ) )
  {
    return false;
  }
  RECT    RC1 = pEntity->CollisionRect();
  RECT    RC2 = pEntity2->CollisionRect();

  RECT    rcTemp;

  return !!IntersectRect( &rcTemp, &RC1, &RC2 );

}



bool Level::Move( Entity* pEntity, int DX, int DY, int DirFlags )
{

  pEntity->m_X += DX;
  pEntity->m_Y += DY;

  // modify carried objects direction
  if ( DirFlags & Dobbs::DIR_DIRECTED_MOVE )
  {
    if ( DirFlags & ( Dobbs::DIR_LEFT | Dobbs::DIR_RIGHT ) )
    {
      pEntity->SetDirection( (Dobbs::DirFlags)( DirFlags & ( Dobbs::DIR_LEFT | Dobbs::DIR_RIGHT ) ) );
    }
  }

  // move any carried entities with it
  std::list<Entity*>::iterator    itCE( pEntity->m_CarriedEntities.begin() );
  while ( itCE != pEntity->m_CarriedEntities.end() )
  {
    Entity*   pMovedEntity( *itCE );
    //if ( !Move( pMovedEntity, DX, DY, pEntity->MoveCarriedObjectFlags() | DirFlags ) )
    if ( pMovedEntity->Move( *this, (float)DX, (float)DY, pEntity->MoveCarriedObjectFlags() | DirFlags ) != Dobbs::BLOCKED_NONE )
    {
      pMovedEntity->OnEvent( Entity::EE_BLOCKED_CARRIED_MOVE, 0, 0, std::string(), pEntity );
    }

    ++itCE;
  }

  if ( ( pEntity->GetType() == Dobbs::ENTITY_TYPE_PLAYER )
  &&   ( !pEntity->IsHidden() ) )
  {
    if ( IsAreaDeadly( pEntity, pEntity->CollisionRect(), DirFlags ) )
    {
      pEntity->Die();
    }
  }
  return true;

}



Entity* Level::FindEntityAt( int X, int Y )
{

  POINT   pt;

  pt.x = X;
  pt.y = Y;
  std::list<Entity*>::iterator      it( m_Entities.begin() );
  while ( it != m_Entities.end() )
  {
    Entity*  pEntity( *it );

    RECT    CollRect = pEntity->CollisionRect();

    if ( PtInRect( &CollRect, pt ) )
    {
      return pEntity;
    }
    ++it;
  }
  return NULL;

}



Entity* Level::FindEntityByType( const Dobbs::EntityTypes EType )
{

  std::list<Entity*>::iterator      it( m_Entities.begin() );
  while ( it != m_Entities.end() )
  {
    Entity*  pEntity( *it );

    if ( pEntity->GetType() == EType )
    {
      return pEntity;
    }
    ++it;
  }
  return NULL;

}



void Level::RemoveEntity( Entity* pEntity )
{

  m_Entities.remove( pEntity );
  delete pEntity;

}



Dobbs::TileType Level::TileBehindEntity( Entity* pEntity, int DX, int DY )
{

  Dobbs::TileType   TT = Tile( pEntity->GetX() / 32 + DX, pEntity->GetY() / 32 + DY );

  if ( TT == Dobbs::TILE_EMPTY )
  {
    return TileFromLayer( 0, pEntity->GetX() / 32 + DX, pEntity->GetY() / 32 + DY );
  }
  return TT;

}



void Level::EnableLayer( int Layer, bool Enable )
{

  if ( ( Layer >= 0 )
  &&   ( Layer <= 2 ) )
  {
    m_DisplayLayer[Layer] = Enable;
  }

}



bool Level::LayerEnabled( int Layer )
{

  if ( ( Layer >= 0 )
  &&   ( Layer <= 2 ) )
  {
    return m_DisplayLayer[Layer];
  }
  return false;

}



EntityParticle* Level::SpawnParticle( Dobbs::ParticleType Type, int X, int Y, unsigned int Color )
{

  EntityParticle*   pParticle = (EntityParticle*)SpawnEntity( Dobbs::ENTITY_TYPE_PARTICLE_EXPLOSION, 0, X, Y );

  pParticle->SetParticleType( Type );
  if ( Color != -1 )
  {
    pParticle->SetColor( Color );
  }

  return pParticle;

}



void Level::ReplaceTile( Dobbs::TileType Tile1, Dobbs::TileType Tile2 )
{

  for ( int i = 0; i < Width(); ++i )
  {
    for ( int j = 0; j < Height(); ++j )
    {
      if ( Tile( i, j ) == Tile1 )
      {
        SetTile( 1, i, j, Tile2 );
      }
    }
  }

}



